// This MFC Samples source code demonstrates using MFC Microsoft Office Fluent User Interface 
// (the "Fluent UI") and is provided only as referential material to supplement the 
// Microsoft Foundation Classes Reference and related electronic documentation 
// included with the MFC C++ library software.  
// License terms to copy, use or distribute the Fluent UI are available separately.  
// To learn more about our Fluent UI licensing program, please visit 
// http://msdn.microsoft.com/officeui.
//
// Copyright (C) Microsoft Corporation
// All rights reserved.

// bbDemoDoc.cpp : implementation of the CbbDemoDoc class
//
#include "stdafx.h"
#include "bbDemo.h"
#include "AudioStream.h"
#include "DiskController.h"

#include "bbDemoDoc.h"
#include "bbDemoView.h"

#include "bb_api.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#endif

const char *xmlString = 
	"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
	"<BB60>\n"
	"	<fileName> </fileName>\n"
	"	<settings>\n"
	"		<centerFreq> 98.0e6 </centerFreq>\n"
	"		<span> 20.0e6 </span>\n"
	"		<referenceLevel> 0.0 </referenceLevel>\n"
	"		<attenuator> 0.0 </attenuator>\n"
	"	</settings>\n"
	"</BB60>\n";

static void msg(const char* c)
{
	MessageBoxA(
		0,
		c,
		_T("Error"),
		MB_OK | MB_ICONWARNING | MB_TOPMOST);
}

static float lerp(float f1, float f2, float d) {
	return f1*(1.0-d) + f2*d;
}

extern volatile int waveFreeBlockCount;

IMPLEMENT_DYNCREATE(CbbDemoDoc, CDocument)

BEGIN_MESSAGE_MAP(CbbDemoDoc, CDocument)
	ON_COMMAND(ID_FILEI_SAVETODISK, &CbbDemoDoc::OnFileSaveToDisk)
	ON_COMMAND(ID_FILEI_STOPSAVING, &CbbDemoDoc::OnFileStopSaving)
	ON_COMMAND(ID_FILEI_FROMDISK, &CbbDemoDoc::OnFileFromDisk)
	ON_COMMAND(ID_FILEI_STOPREADINGFROMDISK, &CbbDemoDoc::OnFileStopReadingFromDisk)
	ON_UPDATE_COMMAND_UI(ID_FILEI_STOPREADINGFROMDISK, &CbbDemoDoc::OnUpdateFileStopReadingFromDisk)
	ON_UPDATE_COMMAND_UI(ID_FILEI_FROMDISK, &CbbDemoDoc::OnUpdateFileFromDisk)
	ON_UPDATE_COMMAND_UI(ID_FILEI_STOPSAVING, &CbbDemoDoc::OnUpdateFileStopSaving)
	ON_UPDATE_COMMAND_UI(ID_FILEI_SAVETODISK, &CbbDemoDoc::OnUpdateFileSaveToDisk)
	ON_COMMAND(ID_FILE_CONNECTDEVICE, &CbbDemoDoc::OnFileConnectdevice)
	ON_COMMAND(ID_FILE_DISCONNECTDEVICE, &CbbDemoDoc::OnFileDisconnectdevice)
END_MESSAGE_MAP()

//
// Constructor, limit time here
//
CbbDemoDoc::CbbDemoDoc()
{
	ippStaticInit();

	mode = IDLE;
	view = NULL;
	buf = NULL;
	fftSpec = NULL;
	leftFreq = 97.1e6f;
	rightFreq = 97.1e6f;

	index = 0;
	realStart = 0;

	as = new AudioStream();
	diskCtrl = new DiskController();
	as->setChannelFreq(leftFreq, LEFT);
	as->setChannelFreq(rightFreq, RIGHT);
	lastChanged = LEFT;

	timeBeginPeriod(1);
	resetPlayback();
}

// 
// Deconstructor
//
CbbDemoDoc::~CbbDemoDoc()
{
	delete as;
	delete diskCtrl;

	timeEndPeriod(1);
}

//
// Where we do most of our contruction
//
BOOL CbbDemoDoc::OnNewDocument()
{
	if (!CDocument::OnNewDocument())
		return FALSE;

	if(!ConnectDevice()) {
		msg("Unable to find device, use file menu to connect device, use"
			" fileIO menu to read from disk");	
		mode = IDLE;
	} else {
		ConfigureDevice();
		mode = FROMDEV; // Device opened
	}

	programRunning = true;

	procThread = AfxBeginThread(
		threadEntry,
		this,
		THREAD_PRIORITY_HIGHEST,
		0,
		CREATE_SUSPENDED);
	procThread->m_bAutoDelete = false;
	procThread->ResumeThread();
	procThread->GetMainWnd();

	return TRUE;
}

//
// Stop the thread
//
void CbbDemoDoc::OnCloseDocument()
{
	DWORD result;
	programRunning = false;
	mode = IDLE;
	
	result = WaitForSingleObject(procThread->m_hThread, 1000);

	if(result == WAIT_TIMEOUT || result == WAIT_FAILED)
		TerminateThread(procThread->m_hThread, 0);

	delete procThread;

	// Close Device
	DisconnectDevice();

	CDocument::OnCloseDocument();
}

//
// Connect the device
//
bool CbbDemoDoc::ConnectDevice()
{
	// Open device
	bbStatus status;
	status = bbOpenDevice(&deviceNum);
	
	// See if opened
	if(status != bbNoError)
		return false;

	deviceOpen = true;
	return true;
}

// 
// Disconnect the device
//
void CbbDemoDoc::DisconnectDevice()
{
	mode = IDLE;
	deviceOpen = false;

	Sleep(10); 

	bbCloseDevice(deviceNum);
}

//
// Configure Device
//
void CbbDemoDoc::ConfigureDevice()
{
	bbStatus status;

	// Configure device
	bbConfigureLevel(deviceNum,	0.0, 30.0);
	bbConfigureCenterSpan(deviceNum, 98.0e6, 20.0e6);
	bbConfigureGain(deviceNum, 2);

	// Initiate raw mode
	status = bbInitiate(deviceNum, BB_STREAMING, BB_STREAM_IF); 

	// Handle error code
	if(status != bbNoError) {
		MessageBoxA(
			0,
			bbGetErrorString(status), // Our error-to-string routine
			_T("Error"),
			MB_OK | MB_ICONWARNING | MB_TOPMOST);
	}
}

//
// Stop the device from streaming
//
void CbbDemoDoc::StopDevice()
{
	bbAbort(deviceNum);
}

//
// Invalidate the view from our thread
// Cannot call UpdateAllViews() from thread
//
void CbbDemoDoc::invalidateView()
{
	POSITION pos = GetFirstViewPosition();
	CbbDemoView* view = (CbbDemoView*)GetNextView(pos);
	view->Invalidate();
}

//
// Static thread entry function, redirects to a class func
//
UINT CbbDemoDoc::threadEntry(LPVOID param)
{
	CbbDemoDoc* doc = (CbbDemoDoc*)param;
	return doc->processThread();
}

//
// Main Thread interfacing with the bb60 device
// Initializes the device, starts streaming,
// Infinite while loop until program ends
//
int CbbDemoDoc::processThread()
{
	static int cc = 0;      // Counter for updating the view, 30hz
	//static int toDiskCounter = 0; // dbg
	
	// Allocate memory for our buffers
	view = ippsMalloc_32f(FFTDIV4);
	buf = ippsMalloc_32f(RETSIZE);
	window = ippsMalloc_32f(FFTSIZE);
	compBuf = ippsMalloc_32f(FFTSIZE);
	diskio = ippsMalloc_16s(RETSIZE);

	// Create window
	buildWindow();

	// Create fft "spec" for ipps library
	ippsFFTInitAlloc_R_32f( 
		&fftSpec, 
		13,
		IPP_FFT_DIV_FWD_BY_N, 
		ippAlgHintFast);

	// Main processing loop
	while(programRunning)
	{
		// Actions dealing with the device
		if(mode == FROMDEV || mode == TODISK) {
			bbFetchRaw(deviceNum, buf, 0);

			// Pass to disk if saving
			if(mode == TODISK) {
				// Turn floats [-1.0, 1.0] to shorts [-32768, 32768]
				bbConvert_32f16s(buf, diskio, 15, RETSIZE);

				diskCtrl->buffer(diskio);
			} 
		}

		// If reading from disk, go to special loop
		if(mode == FROMDISK) {
			getFromDisk();
		}

		// Buffer the data with the audio stream class
		if(mode != IDLE)
			as->bufferData(buf);	

		// Every 1/30th second, do FFT and display spectrum
		if(mode != IDLE) {
			if(cc >= 8) {	
				spectrumAnalysis();
				invalidateView();
				cc = 0;
			} else {
				cc++;
			}
		}

		// No busy loops
		if(mode == IDLE) {
			Sleep(4);
		}
	}

	// End streaming session
	bbAbort(deviceNum);

	// Deallocate our sweep memory
	ippsFFTFree_R_32f(fftSpec);
	ippsFree(view);
	ippsFree(buf);
	ippsFree(window);
	ippsFree(compBuf);
	ippsFree(diskio);

	return 0;
}

//
// The main from disk loop
// Must maintain a certain speed,
// If at any time we run out of data, we need to return to 
//   either idle or device streaming
//
void CbbDemoDoc::getFromDisk()
{
	if(playbackFirstTime) {
		playbackStart = timeGetTime();
		playbackFirstTime = false;
	}

	// Get data from disk
	if(!diskCtrl->get(diskio)) {
		diskCtrl->stop();
		resetPlayback();
		
		// Done streaming from disk, if device was open, re-configure it
		//   other wise, idle
		if(deviceOpen) {
			ConfigureDevice();
			mode = FROMDEV;
		} else {
			mode = IDLE;
		}

		return;
	}

	// Scale the shorts retrieved to float -1.0 to 1.0
	//for(int i = 0; i < RETSIZE; i++) {
	//	buf[i] = (float)diskio[i]*0.000030517f;
	//}
	bbConvert_16s32f(diskio, buf, -15, RETSIZE);

	// Here we slow down read speed if needed
	playbackDelay++;
	int sleepTime = 20 - waveFreeBlockCount;
	if(sleepTime > 10) sleepTime = 10;
	Sleep(sleepTime);

	// Marked for deletion
	if(playbackDelay >= 15) {
		//cout << waveFreeBlockCount << " ";//"Sleeping\n";
		playbackDelay = 0;
	}
}

//
// Reset all values needed for fluid playback
//
void CbbDemoDoc::resetPlayback()
{
	playbackFirstTime = true;
	playbackDelay = 0;
}

// 
// Build a nutall window
//
void CbbDemoDoc::buildWindow()
{
	double windowAvg = 0.0,
		   invAvg = 0.0;

	for(int i = 0; i < FFTSIZE; i++) {
		window[i] = 0.355768 
			- 0.487396*cos(2*PI*i/(FFTSIZE-1)) 
			+ 0.144232*cos(4*PI*i/(FFTSIZE-1)) 
			- 0.012604*cos(6*PI*i/(FFTSIZE-1));	
	}

	// sum up m_window data
	for(int i = 0; i < FFTSIZE; i++) {
		windowAvg += window[i];
	}

	// normalize window
	invAvg = FFTSIZE / windowAvg;
	for(int i = 0; i < FFTSIZE; i++) 
		window[i] *= invAvg;
}

//
// Does an FFT of size 8192, convert to dbm units
//   Put results in the view buffer, update view
//
void CbbDemoDoc::spectrumAnalysis()
{
	// Mult by window, Do FFT 
	ippsMul_32f_I(window, buf, FFTSIZE);
	ippsFFTFwd_RToPerm_32f(buf, compBuf, fftSpec, 0);
	// Convert to amp
	ippsPowerSpectr_32fc(
		(Ipp32fc*)&compBuf[FFTDIV4 + index*4], // 4 instead of 2 because we doubled the fftSize
		view,
		FFTDIV4);
	// 10*log(10)
	ippsLn_32f_I(view, FFTDIV4);
	ippsMulC_32f_I(4.3429448f, view, FFTDIV4);
}

//
// SetStation
//
void CbbDemoDoc::setStation(Channel c, float freq)
{
	if(c == LEFT) {
		leftFreq = freq;
	} else {
		rightFreq = freq;
	}

	lastChanged = c;
	as->setChannelFreq(freq, c);
}

//
// SlideStation, move the last station 200kHz in a direction 'd'
//
void CbbDemoDoc::slideStation(short d)
{
	float f = (lastChanged == LEFT) ? leftFreq : rightFreq;
	f += (d > 0) ? 200.0e3f : -200.0e3f;
	setStation(lastChanged, f);
}

//
// Serialization, not using
//
void CbbDemoDoc::Serialize(CArchive& ar)
{
	if (ar.IsStoring())
	{
		// TODO: add storing code here
	}
	else
	{
		// TODO: add loading code here
	}
}


// CbbDemoDoc diagnostics

#ifdef _DEBUG
void CbbDemoDoc::AssertValid() const
{
	CDocument::AssertValid();
}

void CbbDemoDoc::Dump(CDumpContext& dc) const
{
	CDocument::Dump(dc);
}
#endif //_DEBUG

// CbbDemoDoc commands

//
// Menu option to begin saving
// User chooses an xml file
// The xml file stores common settings, then the file
//  name is used to create numbered binary byte dumps
//
void CbbDemoDoc::OnFileSaveToDisk()
{
	CString path;
	CString xmlPath;
	char drive[MAX_PATH], dir[MAX_PATH], fname[MAX_PATH], ext[MAX_PATH];

	// Lets get the starting file
	char fFilter[] = "XML Files (*.xml)|*.xml||";
	CFileDialog cfd(
		FALSE,
		0,
		0,
		OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT,
		fFilter);

	if(cfd.DoModal() != IDOK) {
		return;
	}

	path = cfd.GetPathName();
	_splitpath(path.GetBuffer(), drive, dir, fname, ext);
	path.Format("%s%s%s", drive, dir, fname);

	xmlPath = path + ".xml";

	diskCtrl->setPath(path.GetBuffer());

	// If storing sets up properly, create the xml file
	if(diskCtrl->beginStoring()) {
		mode = TODISK;
		CopyFileA("settingsExample.xml", xmlPath, FALSE);
		FILE *f = fopen(xmlPath, "w");
		fprintf(f, "%s", xmlString);
	}	
}

//
// Stop saving
//
void CbbDemoDoc::OnFileStopSaving()
{
	if(mode == TODISK) {
		diskCtrl->stop();
		mode = FROMDEV;
	}
}

//
// Select the xml file associated with a file dump
// Attempt to open it
// If exists, set mode, begin reading from file
//
void CbbDemoDoc::OnFileFromDisk()
{
	CString path;
	char drive[MAX_PATH], dir[MAX_PATH], fname[MAX_PATH], ext[MAX_PATH];

	// Lets get the starting file
	char fFilter[] = "XML Files (*.xml)|*.xml||";
	CFileDialog cfd(
		TRUE,
		0,
		0,
		OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT,
		fFilter);
	cfd.DoModal();

	path = cfd.GetPathName();
	_splitpath(path.GetBuffer(), drive, dir, fname, ext);
	path.Format("%s%s%s", drive, dir, fname);

	diskCtrl->setPath(path.GetBuffer());

	if(diskCtrl->beginReading()) {
		StopDevice();
		mode = FROMDISK;
	}
}

//
// Stops reading from disk
// Goes back to getting data from the device
//
void CbbDemoDoc::OnFileStopReadingFromDisk()
{
	if(mode == FROMDISK) {
		diskCtrl->stop();
		
		// Done reading, set device to appropriate set
		if(deviceOpen) {
			ConfigureDevice();
			mode = FROMDEV;
		} else {
			mode = IDLE;
		}

		resetPlayback();
	}
}

//
// ONUPDATE: Menu option to stop playback from a file
//
void CbbDemoDoc::OnUpdateFileStopReadingFromDisk(CCmdUI *pCmdUI)
{
	if(mode == FROMDISK)
		pCmdUI->Enable(1);
	else
		pCmdUI->Enable(0);
}

//
// ONUPDATE: Begin file playback
// Only available when NOT recording or already in playback
//
void CbbDemoDoc::OnUpdateFileFromDisk(CCmdUI *pCmdUI)
{
	if(mode == FROMDISK)
		pCmdUI->SetCheck(1);
	else 
		pCmdUI->SetCheck(0);

	if(mode == FROMDISK || mode == TODISK)
		pCmdUI->Enable(0);
	else
		pCmdUI->Enable(1);
}

//
// ONUPDATE: Stop saving menu option
//
void CbbDemoDoc::OnUpdateFileStopSaving(CCmdUI *pCmdUI)
{
	if(mode == TODISK)
		pCmdUI->Enable(1);
	else
		pCmdUI->Enable(0);
}

//
// ONUPDATE: Begin saving to disk file option
//
void CbbDemoDoc::OnUpdateFileSaveToDisk(CCmdUI *pCmdUI)
{
	if(mode == FROMDEV)
		pCmdUI->Enable(1);
	else
		pCmdUI->Enable(0);

	if(mode == TODISK)
		pCmdUI->SetCheck(1);
	else
		pCmdUI->SetCheck(0);
}

//
// File menu connect device
//
void CbbDemoDoc::OnFileConnectdevice()
{
	if(deviceOpen) {
		return;
	}

	if(mode == FROMDISK) {
		return;
	}

	if(ConnectDevice())
		mode = FROMDEV;
	else
		mode = IDLE;
}

//
// File menu disconnect device
//
void CbbDemoDoc::OnFileDisconnectdevice()
{
	if(!deviceOpen) {
		return;
	}

	if(mode == TODISK) {
		return;
	}

	DisconnectDevice();
}
